"""
compute_Umu.py
~~~~~~~~~~~~~~~

Exponentiate the gauge potential arrays to obtain group‑valued link variables
``U_μ(i)`` for each gauge group.  For the U(1) theory the exponential is
performed on scalars, while for SU(2) and SU(3) the potentials are diagonal
matrices and the exponential reduces to exponentiating the diagonal entries.

The input arrays ``A_U1.npy``, ``A_SU2.npy`` and ``A_SU3.npy`` must have been
produced by ``compute_Amu.py``.  The resulting link variables are saved as
``U_U1.npy``, ``U_SU2.npy`` and ``U_SU3.npy`` in the same ``data_dir``.
"""

from __future__ import annotations

import os
import yaml
import numpy as np


def main(config_path: str = 'config.yaml') -> None:
    """
    Entry point for exponentiating ``A_μ`` into ``U_μ``.

    Parameters
    ----------
    config_path : str
        Path to the configuration YAML file.  Used to resolve the data
        directory relative to the config file location.
    """
    # Resolve config file
    cfg_file = config_path if os.path.isabs(config_path) else os.path.abspath(config_path)
    if not os.path.exists(cfg_file):
        raise FileNotFoundError(f"Cannot find configuration file: {config_path}")
    with open(cfg_file) as f:
        cfg = yaml.safe_load(f)
    base_dir = os.path.dirname(cfg_file)
    data_dir_cfg = cfg.get('data_dir', 'data')
    data_dir = data_dir_cfg if os.path.isabs(data_dir_cfg) else os.path.join(base_dir, data_dir_cfg)
    os.makedirs(data_dir, exist_ok=True)

    # U1
    path_A_u1 = os.path.join(data_dir, 'A_U1.npy')
    if os.path.exists(path_A_u1):
        A_u1 = np.load(path_A_u1, allow_pickle=True)
        # Exponentiate scalar potential: U = exp(i A)
        U_u1 = np.exp(1j * A_u1.astype(float))
        np.save(os.path.join(data_dir, 'U_U1.npy'), U_u1)

    # SU2
    path_A_su2 = os.path.join(data_dir, 'A_SU2.npy')
    if os.path.exists(path_A_su2):
        A_su2 = np.load(path_A_su2, allow_pickle=True)
        # A_su2 shape (N, 2, 2).  It is diagonal, exponentiate each diag element
        n_links = A_su2.shape[0]
        U_su2 = np.zeros_like(A_su2, dtype=complex)
        for i in range(n_links):
            diag_vals = np.diagonal(A_su2[i])
            # complex exponent for each diagonal entry
            diag_exps = np.exp(1j * diag_vals)
            U_su2[i] = np.diag(diag_exps)
        np.save(os.path.join(data_dir, 'U_SU2.npy'), U_su2)

    # SU3
    path_A_su3 = os.path.join(data_dir, 'A_SU3.npy')
    if os.path.exists(path_A_su3):
        A_su3 = np.load(path_A_su3, allow_pickle=True)
        # A_su3 shape (N, 3, 3).  Also diagonal.  Exponentiate diagonal entries
        n_links = A_su3.shape[0]
        U_su3 = np.zeros_like(A_su3, dtype=complex)
        for i in range(n_links):
            diag_vals = np.diagonal(A_su3[i])
            diag_exps = np.exp(1j * diag_vals)
            U_su3[i] = np.diag(diag_exps)
        np.save(os.path.join(data_dir, 'U_SU3.npy'), U_su3)
    return None


if __name__ == '__main__':
    import sys
    cfg = sys.argv[1] if len(sys.argv) > 1 else 'config.yaml'
    main(cfg)